// THUHandler.h - Inteface for the CTHUHandler class
//	Handles THU files and can parse the contents
// Kevin Kirst
// kkirst@fullsail.edu

#ifndef _THUHANDLER_H_
#define _THUHANDLER_H_

#include <fstream>
#include <vector>

typedef unsigned long DWORD;

class CTHUHandler
{
public:
	// Struct THUHeader - The main file header [Size = 16 bytes]
	typedef struct _THUHEADER
	{
		char MagicValue[4];		// Identifier "THU4"
		DWORD dwTextureCount;	// Number of packed textures
		DWORD dwUnkSecOffset;	// Offset to UnknownSection - Follows the texture headers
		DWORD dwUnkSecSize;		// Size (in bytes) of UnknownSection
	} THUHEADER, *LPTHUHEADER;

	// Struct THUTexture - Represents a texture element in the file [Size = 64 bytes]
	typedef struct _THUTEXTURE
	{
		DWORD dwIdentifier;		// unknown but seems to be shared by all textures
		DWORD dwDataOffset;		// Offset into UnknownSection where data about this texture is stored
		DWORD dwUnknown1;		// Unknown (seems to be 16 or 32, depending on file size)
		DWORD dwUnknown2;		// Unknown (seems to be 16 or 32, depending on file size)
		DWORD dwWidth;			// Width of texture
		DWORD dwHeight;			// Height of texture
		DWORD dwDimMult;		// Dimension multiplier (x3, x4, etc), unsure of correct definition
		DWORD dwPixelFormat;	// Pixel format (14h = 20 = DXT1, 18h = 24 = DXT5), unsure of correct definition
		DWORD dwFileLength;		// texture name length
		char pchFile[28];		// texture name (NOT NULL TERMINATED!)
	} THUTEXTURE, *LPTHUTEXTURE;

	typedef std::vector<THUTEXTURE> TEXTURES;

public:
	// Constructor
	CTHUHandler(void);
	CTHUHandler(char const* filepath);

	// Destructor
	~CTHUHandler(void);

public:
	// Load - Load a THU file
	//
	// In:	filepath	File path of THU file to load
	//		bOverwrite	Set to TRUE to overwrite the file
	//
	// Returns TRUE on success
	bool Load(char const* filepath, bool bOverwrite = false);

	// Unload - Unload the loaded THU file
	void Unload(void);

	// Update - Update the loaded THU file with the edited content contained in the object
	//
	// Returns TRUE on success
	bool Update(void);

	// GetHeader - Retrieve a copy of the THU main file header
	inline THUHEADER const& GetHeader(void) const { return this->m_hHeader; }

	// GetTextureCount - Get the amount of textures in the THU file
	inline int GetTextureCount(void) const { return this->m_hHeader.dwTextureCount; }

	// GetTextures - Retrieve a copy of an array of textures contained in the THU file
	inline TEXTURES const& GetTextures(void) const { return this->m_Textures; }
	inline TEXTURES& GetTextures(void) { return this->m_Textures; }

	// GetTextureDataOffset - Get the offset in the file of the texture data
	inline int GetTextureDataOffset(void) const { return this->m_hHeader.dwUnkSecOffset; }

	// GetTextureData - Get the texture data
	inline char const* GetTextureData(void) const { return this->m_pTextureDataSection; }

	// GetTextureDataSize - Get the size of the texture data
	inline int GetTextureDataSize(void) const { return this->m_hHeader.dwUnkSecSize; }

private:
	// _ParseTHUFile - Physically parse the THU file and extract the content
	void _ParseTHUFile(void);

	// _RewriteTHUFile - Rewrite the THU file using updated info in class object
	void _RewriteTHUFile(void);

private:
	std::fstream m_FileStream;		// THU file stream object

	THUHEADER m_hHeader;			// THU file main header
	TEXTURES m_Textures;			// Array of texture elements in this THU file

	char *m_pTextureDataSection;	// Texture data section (AKA UnknownSection)
};

#endif //_THUHANDLER_H_